//*************************************************************************************************
//
//	Description:
//		Skintest.fx - Basic object shader for The Race. Supports vertex colours, Blinn-style diffuse
//		and specular lighting, shininess maps, tangent space normal maps, multiple light sources
//		(of 3 different types), shadow buffers.
//
//	<P> Copyright (c) 2006 Blimey! Games Ltd. All rights reserved.
//
//	Author: 
//		Tom Nettleship
//
//	History:
//
//	<TABLE>
//		\Author         Date        Version       Description
//		--------        -----       --------      ------------
//		TNettleship     04/05/2006  0.1           Created
//		TNettleship     05/12/2006  1.0           First "offical" release to FTP
//		TNettleship     05/18/2006  1.1           Added technique annotations.
//		TNettleship     05/31/2006  1.2           Added renderer lighting interface.
//		TNettleship     05/31/2006  1.3           Added specular overlighting to game engine code.
//		TNettleship     06/08/2006  1.4           Added ambient occlusion term from vertex colour,
//																							global emap factor, added option for compressed
//																							normals.
//		TNettleship			08/01/2006	1.5						Added option for ambient occlusion to affect
//																							diffuse and specular lighting. Fixed a bug with
//																							channel assignment & vertex colours in 3DSMAX.
//		TNettleship			10/16/2006	1.6						Changed the way lighting distance attenuation works.
//		TNettleship			10/19/2006	1.7						Removed texture mapping info params from display,
//																							reverted to parser 0x0000 until the texture management
//																							bugs in 0x0001 can be fixed.
//		TNettleship			11/02/2006	1.8						Fixed a bug with directional lighting.
//		TMann						12/14/2006	1.9						GL & PS3 versions.
//		TMann						12/18/2006	1.10					Changed r/c major for GL & PS3 versions
//		TNettleship			02/12/2007	1.11					Fixed environment mapping
//		TNettleship			04/19/2007	2.0						Added shader specialisation.
//		TNettleship			05/03/2007	2.01					Added override technique for shadow generation mode
//		TNettleship			07/11/2007	2.02					Changed lighting to work in world coords.
//		TNettleship     07/24/2007  2.03          Made sure samplers aren't using anisotropic filtering.
//		TNettleship     07/31/2007  2.04          Added an alphatest technique.
//		TNettleship     08/17/2007  2.05					Removed texture bias.
//		TNettleship			10/23/2007	2.06					Converted to on-load rendermode behaviour binding.
//	<TABLE>
//
//*************************************************************************************************

#define _SSAO_READY_

#include "stddefs.fxh"
#include "specialisation_globals.fxh"


//-----------------------------------------------------------------------
//
// Preprocessor definitions
//

#define MAX_ENVMAP_BIAS	6.0f

// Skinning
#define MAX_BONES 72


#ifdef _XBOX
#define BONES_FROM_VFETCH
#endif


//-----------------------------------------------------------------------
//
// Input parameters
//

//
// Camera
//
#ifdef _3DSMAX_
// 3DSMax parser 0x0001 doesn't support WorldCameraPosition, so we need to bring the view matrix
// in to access the 4th row to get the same information. Parser 0x0000 supports it. Bleh.
float4x4 viewI : ViewInverse
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
// The ingame renderer directly supplies the camera position
SHARE_PARAM float3 worldCameraPos : WorldCameraPosition
<
	string UIWidget = "None";
	bool appEdit = false;
>;
#endif



//
// Transforms
//

#if defined( _3DSMAX_ ) || defined(USE_WVP_CONSTANT)
// Max doesn't support viewproj as an app-supplied parameter
float4x4 worldviewproj : WorldViewProjection
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
SHARE_PARAM float4x4 viewproj : ViewProjection
<
	bool appEdit = false;
	bool export = false;
>;
#endif

float4x4 world : World
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;

#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
float4x4 worldI : WorldInverse
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;
#endif


//
// Channel mappings (max only)
//

//
// N.B. Max contains a bug which means the colour channel must NOT be mapped to texcoord0.
// The first UV coord channel MUST be mapped to texcoord0 or the basis vectors for normal
// mapping will be screwed up. (e.g. there's some bit of code deep within max which assumes
// this setup when calculating the basis vectors)
//

#ifdef _3DSMAX_

// First UV channel
int texcoord0 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 0;
	int MapChannel = 1;
	int RuntimeTexcoord = 0;
	bool export = false;
> = 0;

// Vertex colour channel
int texcoord1 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 1;
	int MapChannel = 0;
	bool ColorChannel = true;
	bool export = false;
> = 0;

#endif


//
// Textures
//

#ifdef _3DSMAX_
texture diffuseTexture : DiffuseMap						// Diffuse colour in RGB, translucency in alpha
#else
texture diffuseTexture : TEXTURE							// Diffuse colour in RGB, translucency in alpha
#endif
<
	string UIName = "Diffuse Texture";
	bool appEdit = true;
>;

SPECIALISATION_PARAM_DEFAULTS_TRUE( useSpecular, "Use specular?", "USE_SPECULAR" )	// TRUE if the specular lighting is to be used
#if defined( _3DSMAX_ ) || defined( USE_SPECULAR )
DEPENDENT_TEXTURE_PARAM( specularTexture, "Spec Tex {UV1}", useSpecular )
#endif

SPECIALISATION_PARAM( useNormalMap, "Use normal map?", "NORMAL_MAPPING" )	// TRUE if the normal map is to be used in lighting
DECLARE_DEPENDENT_VERTEX_STREAM( tangentDependency, tangent, TANGENT, useNormalMap )
DECLARE_DEPENDENT_VERTEX_STREAM( binormalDependency, binormal, BINORMAL, useNormalMap )

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
DEPENDENT_TEXTURE_PARAM( normalTexture, "Normal Texture", useNormalMap )
#endif

SPECIALISATION_PARAM( useEnvironmentMap, "Use environment map?", "ENV_MAPPING" )	// TRUE if the environment map is to be used

#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
DEPENDENT_CUBE_TEXTURE_PARAM( environmentTexture, "Env Texture", useEnvironmentMap, false )
#endif

#if defined( _3DSMAX_ ) || defined( USE_SPECULAR )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( minSpecPower, "Min Specular Power", useSpecular, 1.0f, 1024.0f, 1.0f )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( maxSpecPower, "Max Specular Power", useSpecular, 1.0f, 1024.0f, 32.0f )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( globalSpecularFactor, "Specular Factor", useSpecular, 0.0f, 10.0f, 1.0f )
#endif

#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( globalEMapFactor, "EMap Factor", useEnvironmentMap, 0.0f, 16.0f, 1.0f )
#endif

SPECIALISATION_PARAM( useScrolling, "Enable UV scrolling?", "UV_SCROLL" ) // TRUE if UV scrolling is enabled
#if defined( _3DSMAX_ ) || defined( UV_SCROLL )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( scrollUspeed, "Scroll speed in U", useScrolling, -16.0f, 16.0f, 1.0f )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( scrollVspeed, "Scroll speed in V", useScrolling, -16.0f, 16.0f, 0.0f )
#endif

SPECIALISATION_PARAM( useJumping, "Enable UV jumping?", "UV_JUMP" ) // TRUE if UV jumping is enabled
#if defined( _3DSMAX_ ) || defined( UV_JUMP )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( numCellsU, "number of cells in U", useJumping, 1.0f, 16.0f, 4.0f )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( numCellsV, "number of cells in V", useJumping, 1.0f, 16.0f, 1.0f )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( jumpSpeed, "speed scale for jumps", useJumping, 0.0f, 16.0f, 1.0f )
#endif

#if defined( _3DSMAX_ ) || defined( UV_SCROLL ) || defined( UV_JUMP )
float currentTime : TIME
<
> = 0.0f;
#endif

#if defined( _3DSMAX_ ) 
#define SPEED_SCALE	0.0625f	// one sixteenth
#else
#define SPEED_SCALE 0.25f // one quarter - four times faster in game
#endif


// Simplify shadowing code to access only the first (the nearest) shadowmap.
SPECIALISATION_PARAM( forceFistShadowmap, "Force the first shadowmap?", "FORCE_FIRST_SHADOWMAP" )


//
// Lighting
//

#include "lighting_globals.fxh"
DECLARE_LIGHTING_PARAMS

//
// This shader supports variance 3 mask tint
//
#include "variance_3mask_tint.fxh"


// colour multiplier, forced to end to avoid artists touching it

float4 globalColourMultiplier
<
	string UIWidget = "None";
	bool appEdit = true;
	bool export = true;
> = { 1.0f, 1.0f, 1.0f, 1.0f };


//-----------------------------------------------------------------------
//
// Samplers
//

sampler2D diffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="diffuseTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < diffuseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

#if defined( _3DSMAX_ ) || defined( USE_SPECULAR )
sampler2D specularMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="specularTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < specularTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
sampler2D normalMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="normalTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < normalTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif


#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
samplerCUBE environmentMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="environmentTexture";
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Clamp";
	string AddressV  = "Clamp";
	string AddressW  = "Clamp";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < environmentTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
#if defined(_PS3_)
	AddressU  = ClampToEdge;
	AddressV  = ClampToEdge;
	AddressW  = ClampToEdge;
	LODBias = 0;
#else
	AddressU  = Clamp;
	AddressV  = Clamp;
	AddressW  = Clamp;
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif


//-----------------------------------------------------------------------
//
// Bones
//

#ifndef _3DSMAX_
	#if !defined(_PS3_)
		#if defined(BONES_FROM_VFETCH)
			bool usesBonesFromVFetch	// dummy variable for the code to read
			<
				bool appEdit = false;
			>;
		#else
			float4x3 boneTransforms[ MAX_BONES ] : boneTransforms
			<
				bool appEdit = false;
			>;
		#endif
	#endif
#endif



//-----------------------------------------------------------------------
//
// Vertex Shader(s)
//

// Input structures
struct VSINPUT
{
	float3 position : POSITION;														// Object space position
#ifdef _3DSMAX_
	float4 colour   : TEXCOORD1;													// Vertex colour
	float2 texCoord : TEXCOORD0;													// UV channel 1 texture coord - N.B. MAx requires that texcoord0 is a geometric channel
																												// as it implicitly uses that to calculate the tangent space coordinate frame.
#else
	float4 colour   : COLOR0;															// Vertex colour
	float2 texCoord : TEXCOORD0;													// UV channel 1 texture coord
#endif
	float3 normal   : NORMAL;															// Object space normal
#ifndef _PS3_
	float4 boneWeights : BLENDWEIGHT;											// Bone weights
	int4   boneIndices : BLENDINDICES;										// Bone indices
#endif

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
	// These two vertex streams aren't needed without normal_mapping
	float3 tangent  : TANGENT;														// Object space tangent
	float3 binormal : BINORMAL;														// Object space normal
#endif
};


struct SHADOWGEN_VSINPUT
{
	float3 position : POSITION;														// Object space position
#ifndef _PS3_	
	float4 boneWeights : BLENDWEIGHT;											// Bone weights
	int4   boneIndices : BLENDINDICES;										// Bone indices
#endif
};


// Output structures
struct VSOUTPUT
{
	float4 position		: POSITION;													// View-coords position
	float4 colour			: TEXCOORD3;														// Vertex colour
	float2 texCoord		: TEXCOORD0;												// UV coords for texture channel 0
	float3 normal			: TEXCOORD1;												// Normal vector (world space)
	float4 eye				: TEXCOORD2;												// Eye vector (world space)

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
	// These two vertex streams aren't needed without normal_mapping
	float3 tangent		: TEXCOORD4;												// Tangent vector (world space)
	float3 binormal		: TEXCOORD5;												// Normal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS_VS( 6 )
#else
	DECLARE_LIGHTING_INTERPOLATORS_VS( 4 )
#endif
};

struct VSOUTPUT_UNLIT
{
	float4 position		: POSITION;													// View-coords position
	float4 colour		: TEXCOORD3;												// Vertex colour
	float2 texCoord		: TEXCOORD0;												// UV coords for texture channel 0
};

struct VSOUTPUT_LD
{
	float4 position		: POSITION;													// View-coords position
	float4 colour			: TEXCOORD3;														// Vertex colour
	float2 texCoord		: TEXCOORD0;												// UV coords for texture channel 0
#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
	float3 normal			: TEXCOORD1;												// Normal vector (world space)
	float4 eye				: TEXCOORD2;												// Eye vector (world space)
#endif
};


struct SHADOWGEN_VSOUTPUT
{
	float4 position		 : POSITION;												// View-coords position
#ifdef NEEDS_SHADOW_COORDS
	float4 shadowCoord : TEXCOORD0;												// UV coords for texture channel 0
#endif
};

struct ZPRIMEDOF_VSOUTPUT
{
	float4 position		 : POSITION;												// View-coords position
	float4 coords			 : TEXCOORD0;
};


#ifdef BONES_FROM_VFETCH
float4x3 VFetchBoneMatrixTextureCache( int BoneIndex )
{
    float4 Result1;
    float4 Result2;
    float4 Result3;
    asm
    {
        vfetch Result1, BoneIndex, position1, UseTextureCache=true
        vfetch Result2, BoneIndex, position2, UseTextureCache=true
        vfetch Result3, BoneIndex, position3, UseTextureCache=true
    };
    
    float4x3 Result;
    Result._11_21_31_41 = Result1;
    Result._12_22_32_42 = Result2;
    Result._13_23_33_43 = Result3;
    return Result;
}
#endif

//-----------------------------------------------------------------------
//
// Vertex shader code
//
// TODO: rearrange to transform eye and lights into local space before
// vector calculations.

VSOUTPUT SkintestVertexShader( VSINPUT _input )
{
	VSOUTPUT _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	// Perform skinning:
	//	Using ***object space*** bone xforms, xform the position, normal,
	//	tangent and binormal vectors using the weights and indices for
	//	this vertex.
	float3 skinnedPos = 0.0f;
	float3 skinnedNormal = 0.0f;
	DEPENDENT_CODE_START( useNormalMap )
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
		float3 skinnedTangent = 0.0f;
		float3 skinnedBinormal = 0.0f;
#endif
	DEPENDENT_CODE_END( useNormalMap )

//**********************************************************************
//  NOTE : NON-MAX SECTION HERE, PAGE DOWN FOR MAX VERSION
//**********************************************************************

#ifndef _3DSMAX_
#ifndef _PS3_

		// cast the vectors to arrays for use in the for loop below
		float blendWeightsArray[ 4 ] = ( float[ 4 ] )_input.boneWeights;
		
	#ifdef BONES_FROM_VFETCH

    float4x3 LocalToWorldMatrix = 0;
		for ( int boneIndex = 0; boneIndex < 4; boneIndex++ )
		{
			LocalToWorldMatrix += blendWeightsArray[boneIndex] * VFetchBoneMatrixTextureCache( _input.boneIndices[boneIndex] );
		}
		skinnedPos = mul( float4( _input.position, 1.0f ), LocalToWorldMatrix );
		skinnedNormal = mul( _input.normal, (float3x3)LocalToWorldMatrix );
		DEPENDENT_CODE_START( useNormalMap )
		#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
		skinnedTangent = mul( _input.tangent, (float3x3)LocalToWorldMatrix );
		skinnedBinormal = mul( _input.binormal, (float3x3)LocalToWorldMatrix );
		#endif
		DEPENDENT_CODE_END( useNormalMap )

	#else

		float totalWeight = 0.0f;
	
		// calculate the pos/tangent space using the "normal" weights
		for ( int boneIndex = 0; boneIndex < 4; boneIndex++ )
		{
			float weight = blendWeightsArray[ boneIndex ];
			if ( weight > 0.0f )
			{
				totalWeight += weight;

				skinnedPos += mul( float4( _input.position, 1.0f ), boneTransforms[ _input.boneIndices[ boneIndex ] ] ) * weight;
				skinnedNormal += mul( _input.normal, boneTransforms[ _input.boneIndices[ boneIndex ] ] ) * weight;
				DEPENDENT_CODE_START( useNormalMap )
	#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
					skinnedTangent += mul( _input.tangent, boneTransforms[ _input.boneIndices[ boneIndex ] ] ) * weight;
					skinnedBinormal += mul( _input.binormal, boneTransforms[ _input.boneIndices[ boneIndex ] ] ) * weight;
	#endif
				DEPENDENT_CODE_END( useNormalMap )
			}
		}
		totalWeight = 1.0f / totalWeight;

		// Calculate the weighted average of the influences
		skinnedPos *= totalWeight;
		skinnedNormal *= totalWeight;
		DEPENDENT_CODE_START( useNormalMap )
		#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
			skinnedTangent *= totalWeight;
			skinnedBinormal *= totalWeight;
		#endif
		DEPENDENT_CODE_END( useNormalMap )
		
	#endif	// BONES_FROM_VFETCH
	

#else
		skinnedPos=_input.position;
		skinnedNormal=_input.normal;
#endif
		// Calculate clip-space position of the vertex
		_output.position = mul( float4( skinnedPos, 1.0f ), worldviewproj );

		// Calculate vert's world position
		float3 worldPos = mul( float4( skinnedPos, 1.0f ), world ).xyz;

		// Calculate world-space coordinate frame
		float3 normal = normalize( mul( float4( skinnedNormal, 0.0f ), world ).xyz );
		_output.normal = normal;

		DEPENDENT_CODE_START( useNormalMap )
	#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
#ifdef _PS3_	
			skinnedTangent=_input.tangent;
			skinnedBinormal=_input.binormal;
#endif
			_output.tangent  = mul( float4( skinnedTangent, 0.0f ), world ).xyz;
			_output.binormal = mul( float4( skinnedBinormal, 0.0f ), world ).xyz;
	#endif
		DEPENDENT_CODE_END( useNormalMap )

//**********************************************************************
//  MAX VERSION STARTS HERE !
//**********************************************************************
#else

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	// Calculate vert's world position
	float3 worldPos = mul( float4( _input.position, 1.0f ), world ).xyz;

	// Calculate world-space coordinate frame
	float3 normal = normalize( mul( float4( _input.normal, 0.0f ), world ).xyz );
	_output.normal = normal;
	_output.tangent  = mul( float4( _input.tangent, 0.0f ), world ).xyz;
	_output.binormal = mul( float4( _input.binormal, 0.0f ), world ).xyz;
#endif		// #ifndef _3DSMAX_

//**********************************************************************
//  SHARED CODE STARTS HERE !
//**********************************************************************

	// Copy simple invariant params to output structure
	_output.colour = _input.colour;

	DEPENDENT_CODE_START( useScrolling )	
	#if defined( _3DSMAX_ ) || defined( UV_SCROLL )
		{
			_output.texCoord.x = _input.texCoord.x + scrollUspeed * currentTime * SPEED_SCALE;
			_output.texCoord.y = _input.texCoord.y + scrollVspeed * currentTime * SPEED_SCALE;
		}
	#endif
	DEPENDENT_CODE_ELSE( useScrolling )
	#if defined( _3DSMAX_ ) || !defined( UV_SCROLL )
		{
			_output.texCoord = _input.texCoord;
		}
	#endif
	DEPENDENT_CODE_END( useScrolling )

	DEPENDENT_CODE_START( useJumping )	
	#if defined( _3DSMAX_ ) || defined( UV_JUMP )
		{
			float scalex = 1.0f / numCellsU;
			float scaley = 1.0f / numCellsV;
			float offsetx = int(currentTime * numCellsU * jumpSpeed * SPEED_SCALE );								// steps across steadily as time marches on
			float offsety = int(currentTime * numCellsU * jumpSpeed * numCellsV * SPEED_SCALE );	  // steps up as time goes on, going one whole texture size in the time of one step for x
	
	#if defined( _3DSMAX_ ) 			
			_output.texCoord.x += offsetx;
			_output.texCoord.y -= offsety;
	#else
			_output.texCoord.x += offsetx;
			_output.texCoord.y -= (1.0f + offsety);
	#endif
			_output.texCoord.x *= scalex;
			_output.texCoord.y *= scaley;
		}
	#endif
	DEPENDENT_CODE_END( useJumping )

	// Calculate world-space vector to the eye
#ifdef _3DSMAX_
	float3 worldEyeVec = viewI[ 3 ] - worldPos;
#else
	float3 worldEyeVec = worldCameraPos - worldPos;
#endif
	_output.eye = float4(worldEyeVec,0);

#if !defined( _3DSMAX_ ) && !defined( USE_SPECULAR )
	// In the engine, specialisations which have no specular defined need to declare these constants
	// as the lighting macros at the end need them, and they're not declared anywhere else.
	float globalSpecularFactorValue = 0.0f;
	float minSpecPowerValue = 1.0f;
	float maxSpecPowerValue = 1.0f;
#endif

	// Do lighting
	DO_VS_LIGHTING_CALCULATIONS

	_output.colour *= globalColourMultiplier;

	return _output;
}	


///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

VSOUTPUT_UNLIT SkintestUnlitVertexShader( VSINPUT _input )
{
	VSOUTPUT_UNLIT _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	// Perform skinning:
	//	Using ***object space*** bone xforms, xform the position, normal,
	//	tangent and binormal vectors using the weights and indices for
	//	this vertex.
	float3 skinnedPos = 0.0f;
	
#ifndef _3DSMAX_
#ifndef _PS3_

	// cast the vectors to arrays for use in the for loop below
	float blendWeightsArray[ 4 ] = ( float[ 4 ] )_input.boneWeights;

	#ifdef BONES_FROM_VFETCH

    float4x3 LocalToWorldMatrix = 0;
		for ( int boneIndex = 0; boneIndex < 4; boneIndex++ )
		{
			LocalToWorldMatrix += blendWeightsArray[boneIndex] * VFetchBoneMatrixTextureCache( _input.boneIndices[boneIndex] );
		}
		skinnedPos = mul( float4( _input.position, 1.0f ), LocalToWorldMatrix );

	#else

		float totalWeight = 0.0f;

		// calculate the pos/tangent space using the "normal" weights
		for ( int boneIndex = 0; boneIndex < 4; boneIndex++ )
		{
			float weight = blendWeightsArray[ boneIndex ];
			if ( weight > 0.0f )
			{
				totalWeight += weight;

				skinnedPos += mul( float4( _input.position, 1.0f ), boneTransforms[ _input.boneIndices[ boneIndex ] ] ) * weight;
			}
		}
		totalWeight = 1.0f / totalWeight;

		// Calculate the weighted average of the influences
		skinnedPos *= totalWeight;
		
	#endif	// BONES_FROM_VFETCH
		
#else
	skinnedPos=_input.position;
#endif		

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( skinnedPos, 1.0f ), worldviewproj );

#else		// #ifndef _3DSMAX_
	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

#endif		// #ifndef _3DSMAX_

	// Copy simple invariant params to output structure
	_output.colour = _input.colour;
	_output.colour *= globalColourMultiplier;

	DEPENDENT_CODE_START( useScrolling )	
	#if defined( _3DSMAX_ ) || defined( UV_SCROLL )
		{
			_output.texCoord.x = _input.texCoord.x + scrollUspeed * currentTime * SPEED_SCALE;
			_output.texCoord.y = _input.texCoord.y + scrollVspeed * currentTime * SPEED_SCALE;
		}
	#endif
	DEPENDENT_CODE_ELSE( useScrolling )
	#if defined( _3DSMAX_ ) || !defined( UV_SCROLL )
		{
			_output.texCoord = _input.texCoord;
		}
	#endif
	DEPENDENT_CODE_END( useScrolling )

	DEPENDENT_CODE_START( useJumping )	
	#if defined( _3DSMAX_ ) || defined( UV_JUMP )
		{
			float scalex = 1.0f / numCellsU;
			float scaley = 1.0f / numCellsV;
			float offsetx = int(currentTime * numCellsU * jumpSpeed * SPEED_SCALE );								// steps across steadily as time marches on
			float offsety = int(currentTime * numCellsU * jumpSpeed * numCellsV * SPEED_SCALE );	  // steps up as time goes on, going one whole texture size in the time of one step for x
	
	#if defined( _3DSMAX_ ) 			
			_output.texCoord.x += offsetx;
			_output.texCoord.y -= offsety;
	#else
			_output.texCoord.x += offsetx;
			_output.texCoord.y -= (1.0f + offsety);
	#endif
			_output.texCoord.x *= scalex;
			_output.texCoord.y *= scaley;
		}
	#endif
	DEPENDENT_CODE_END( useJumping )


	return _output;
}


/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

VSOUTPUT_LD SkintestLowDetailVertexShader( VSINPUT _input )
{
	VSOUTPUT_LD _output;
	
#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	// Perform skinning:
	//	Using ***object space*** bone xforms, xform the position, normal,
	//	tangent and binormal vectors using the weights and indices for
	//	this vertex.
	float3 skinnedPos = 0.0f;
	float3 skinnedNormal = 0.0f;

#ifndef _3DSMAX_
#ifndef _PS3_

	// cast the vectors to arrays for use in the for loop below
	float blendWeightsArray[ 4 ] = ( float[ 4 ] )_input.boneWeights;

	#ifdef BONES_FROM_VFETCH

		// only do 2 weights
    float4x3 LocalToWorldMatrix = 0;
		float totalweight = blendWeightsArray[0]+blendWeightsArray[1];
		for ( int boneIndex = 0; boneIndex < 2; boneIndex++ )
		{
			LocalToWorldMatrix += (blendWeightsArray[boneIndex]/totalweight) * VFetchBoneMatrixTextureCache( _input.boneIndices[boneIndex] );
		}
		skinnedPos = mul( float4( _input.position, 1.0f ), LocalToWorldMatrix );
		skinnedNormal = mul( float4( _input.normal, 1.0f ), (float3x3)LocalToWorldMatrix );

	#else

		float totalWeight = 0.0f;

		// calculate the pos/tangent space using the "normal" weights
		for ( int boneIndex = 0; boneIndex < 4; boneIndex++ )
		{
			float weight = blendWeightsArray[ boneIndex ];
			if ( weight > 0.0f )
			{
				totalWeight += weight;

				skinnedPos += mul( float4( _input.position, 1.0f ), boneTransforms[ _input.boneIndices[ boneIndex ] ] ) * weight;
				skinnedNormal += mul( _input.normal, boneTransforms[ _input.boneIndices[ boneIndex ] ] ) * weight;
			}
		}
		totalWeight = 1.0f / totalWeight;

		// Calculate the weighted average of the influences
		skinnedPos *= totalWeight;
		skinnedNormal *= totalWeight;
		
	#endif	// BONES_FROM_VFETCH
		
#else
	skinnedPos=_input.position;
	skinnedNormal=_input.normal;
#endif	
	// Calculate clip-space position of the vertex
	_output.position = mul( float4( skinnedPos, 1.0f ), worldviewproj );

	// Calculate vert's world position
	float3 worldPos = mul( float4( skinnedPos, 1.0f ), world ).xyz;

	// Calculate world-space coordinate frame
	float3 normal = normalize( mul( float4( skinnedNormal, 0.0f ), world ).xyz );

#else
	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	// Calculate vert's world position
	float3 worldPos = mul( float4( _input.position, 1.0f ), world ).xyz;

	// Calculate world-space coordinate frame
	float3 normal = normalize( mul( float4( _input.normal, 0.0f ), world ).xyz );
#endif		// #ifndef _3DSMAX_

	// Copy simple invariant params to output structure
	_output.colour = _input.colour;

	DEPENDENT_CODE_START( useScrolling )	
	#if defined( _3DSMAX_ ) || defined( UV_SCROLL )
		{
			_output.texCoord.x = _input.texCoord.x + scrollUspeed * currentTime * SPEED_SCALE;
			_output.texCoord.y = _input.texCoord.y + scrollVspeed * currentTime * SPEED_SCALE;
		}
	#endif
	DEPENDENT_CODE_ELSE( useScrolling )
	#if defined( _3DSMAX_ ) || !defined( UV_SCROLL )
		{
			_output.texCoord = _input.texCoord;
		}
	#endif
	DEPENDENT_CODE_END( useScrolling )

	DEPENDENT_CODE_START( useJumping )	
	#if defined( _3DSMAX_ ) || defined( UV_JUMP )
		{
			float scalex = 1.0f / numCellsU;
			float scaley = 1.0f / numCellsV;
			float offsetx = int(currentTime * numCellsU * jumpSpeed * SPEED_SCALE );								// steps across steadily as time marches on
			float offsety = int(currentTime * numCellsU * jumpSpeed * numCellsV * SPEED_SCALE );	  // steps up as time goes on, going one whole texture size in the time of one step for x
	
	#if defined( _3DSMAX_ ) 			
			_output.texCoord.x += offsetx;
			_output.texCoord.y -= offsety;
	#else
			_output.texCoord.x += offsetx;
			_output.texCoord.y -= (1.0f + offsety);
	#endif
			_output.texCoord.x *= scalex;
			_output.texCoord.y *= scaley;
		}
	#endif
	DEPENDENT_CODE_END( useJumping )

	// Calculate world-space vector to the eye
	DEPENDENT_CODE_START( useEnvironmentMap )
#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
	_output.normal = normal;
	#ifdef _3DSMAX_
		float3 worldEyeVec = viewI[ 3 ] - worldPos;
	#else
		float3 worldEyeVec = worldCameraPos - worldPos;
	#endif
		_output.eye = float4(worldEyeVec,0);
#endif
	DEPENDENT_CODE_END( useEnvironmentMap )


	// Do lighting
	DO_VERTEX_LIGHTING( worldPos, normal, _output.colour )

	_output.colour *= globalColourMultiplier;

	return _output;
}


///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

SHADOWGEN_VSOUTPUT SkintestShadowGenVertexShader( SHADOWGEN_VSINPUT _input )
{
	SHADOWGEN_VSOUTPUT _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	// Perform skinning:
	//	Using ***object space*** bone xforms, xform the position, normal,
	//	tangent and binormal vectors using the weights and indices for
	//	this vertex.
	float3 skinnedPos = 0.0f;

#ifndef _3DSMAX_
#ifndef _PS3_

	// cast the vectors to arrays for use in the for loop below
	float blendWeightsArray[ 4 ] = ( float[ 4 ] )_input.boneWeights;
	
	#ifdef BONES_FROM_VFETCH

    float4x3 LocalToWorldMatrix = 0;
		for ( int boneIndex = 0; boneIndex < 4; boneIndex++ )
		{
			LocalToWorldMatrix += blendWeightsArray[boneIndex] * VFetchBoneMatrixTextureCache( _input.boneIndices[boneIndex] );
		}
		skinnedPos = mul( float4( _input.position, 1.0f ), LocalToWorldMatrix );

	#else
		
		float totalWeight = 0.0f;

		// calculate the pos/tangent space using the "normal" weights
		for ( int boneIndex = 0; boneIndex < 4; boneIndex++ )
		{
			float weight = blendWeightsArray[ boneIndex ];
			if ( weight > 0.0f )
			{
				totalWeight += weight;

				skinnedPos += mul( float4( _input.position, 1.0f ), boneTransforms[ _input.boneIndices[ boneIndex ] ] ) * weight;
			}
		}
		totalWeight = 1.0f / totalWeight;

		// Calculate the weighted average of the influences
		skinnedPos *= totalWeight;
		
	#endif	// BONES_FROM_VFETCH
		
#else
	skinnedPos=_input.position;
#endif	
#endif

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( skinnedPos, 1.0f ), worldviewproj );
#ifdef NEEDS_SHADOW_COORDS
	OUTPUT_SHADOW_COORDS;
#endif

	return _output;
}


ZPRIMEDOF_VSOUTPUT SkintestZPrimeDOFVertexShader( SHADOWGEN_VSINPUT _input )
{
	ZPRIMEDOF_VSOUTPUT _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	// Perform skinning:
	//	Using ***object space*** bone xforms, xform the position, normal,
	//	tangent and binormal vectors using the weights and indices for
	//	this vertex.
	float3 skinnedPos = 0.0f;

#ifndef _3DSMAX_
#ifndef _PS3_

	// cast the vectors to arrays for use in the for loop below
	float blendWeightsArray[ 4 ] = ( float[ 4 ] )_input.boneWeights;
	
	#ifdef BONES_FROM_VFETCH

    float4x3 LocalToWorldMatrix = 0;
		for ( int boneIndex = 0; boneIndex < 4; boneIndex++ )
		{
			LocalToWorldMatrix += blendWeightsArray[boneIndex] * VFetchBoneMatrixTextureCache( _input.boneIndices[boneIndex] );
		}
		skinnedPos = mul( float4( _input.position, 1.0f ), LocalToWorldMatrix );

	#else
		
		float totalWeight = 0.0f;

		// calculate the pos/tangent space using the "normal" weights
		for ( int boneIndex = 0; boneIndex < 4; boneIndex++ )
		{
			float weight = blendWeightsArray[ boneIndex ];
			if ( weight > 0.0f )
			{
				totalWeight += weight;
				skinnedPos += mul( float4( _input.position, 1.0f ), boneTransforms[ _input.boneIndices[ boneIndex ] ] ) * weight;
			}
		}
		totalWeight = 1.0f / totalWeight;

		// Calculate the weighted average of the influences
		skinnedPos *= totalWeight;
		
	#endif	// BONES_FROM_VFETCH
		
#else
	skinnedPos=_input.position;
#endif	
#endif

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( skinnedPos, 1.0f ), worldviewproj );
	_output.coords = _output.position;

	return _output;
}



//-----------------------------------------------------------------------
//
// Fragment Shader(s)
//

#if defined( _3DSMAX_ )
// Max can't handle centroid interpolators properly

// Input structure
struct PSINPUT
{
	float4 colour			: TEXCOORD3;										// Vertex colour
	float2 texCoord		: TEXCOORD0;								// UV coords for texture channel 0
	float3 normal			: TEXCOORD1;								// Normal vector (world space)
	float4 eye				: TEXCOORD2;								// Eye vector (world space)
	float3 tangent		: TEXCOORD4;								// Tangent vector (world space)
	float3 binormal		: TEXCOORD5;								// Normal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS_PS( 6 )
};

struct PSINPUT_UNLIT
{
	float4 colour			: TEXCOORD3;										// Vertex colour
	float2 texCoord		: TEXCOORD0;								// UV coords for texture channel 0
	float3 normal			: TEXCOORD1;								// Normal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS_PS( 4 )
};

struct PSINPUT_LD
{
	float4 colour			: TEXCOORD3;										// Vertex colour
	float2 texCoord		: TEXCOORD0;								// UV coords for texture channel 0
#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
	float3 normal			: TEXCOORD1;								// Normal vector (world space)
	float4 eye				: TEXCOORD2;								// Eye vector (world space)
#endif
};

#else

struct PSINPUT
{
	float4 colour			: TEXCOORD3;														// Vertex colour
	float2 texCoord		: TEXCOORD0;												// UV coords for texture channel 0
	float3 normal			: TEXCOORD1_centroid;								// Normal vector (world space)
	float4 eye				: TEXCOORD2_centroid;								// Eye vector (world space)

#if defined( NORMAL_MAPPING )
	// These two vertex streams aren't needed without normal_mapping
	float3 tangent		: TEXCOORD4_centroid;								// Tangent vector (world space)
	float3 binormal		: TEXCOORD5_centroid;								// Normal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS_PS( 6 )
#else
	DECLARE_LIGHTING_INTERPOLATORS_PS( 4 )
#endif
	DECLARE_SHADOW_PS_INPUTS
};

struct PSINPUT_UNLIT
{
	float4 colour			: TEXCOORD3;										// Vertex colour
	float2 texCoord		: TEXCOORD0;								// UV coords for texture channel 0
	float3 normal			: TEXCOORD1;								// Normal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS_PS( 4 )
};

struct PSINPUT_LD
{
	float4 colour			: TEXCOORD3;														// Vertex colour
	float2 texCoord		: TEXCOORD0;												// UV coords for texture channel 0
#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
	float3 normal			: TEXCOORD1_centroid;								// Normal vector (world space)
	float4 eye				: TEXCOORD2_centroid;								// Eye vector (world space)
#endif
};

#endif


struct SHADOWGEN_PSINPUT
{
#if defined(NEEDS_SHADOW_COORDS) || defined(_PS3_)
	float4 shadowCoord : TEXCOORD0;
#endif
};


// Output structure
struct PSOUTPUT
{
	COLOUR_OUTPUT_TYPE Colour : COLOR0;
};



//-----------------------------------------------------------------------
//
// Fragment shader code
//

REMOVE_UNUSED_INTERPOLATORS
PSOUTPUT SkintestFragmentShader( PSINPUT _input )
{
	PSOUTPUT _output;

	PS_GENERATE_WORLDPOS( _input.eye.xyz )
	
	// Read textures
	float4 diffuseTexColour = Tint_DoIfEnabled( tex2D( diffuseMap, _input.texCoord ), _input.texCoord, false );
	float4 specularTexColour = float4( 0.0f, 0.0f, 0.0f, 0.0f );
	
	float globalSpecularFactorValue;
	float minSpecPowerValue;
	float maxSpecPowerValue;

#if !defined( _3DSMAX_ ) && !defined( USE_SPECULAR )
	// In the engine, specialisations which have no specular defined need to declare these constants
	// as the lighting macros at the end need them, and they're not declared anywhere else.
#else
	DEPENDENT_CODE_START( useSpecular )
	globalSpecularFactorValue = globalSpecularFactor;
	minSpecPowerValue = minSpecPower;
	maxSpecPowerValue = maxSpecPower;
	DEPENDENT_CODE_END( useSpecular )
#endif

	DEPENDENT_CODE_START( useSpecular )
#if defined( _3DSMAX_ ) || defined( USE_SPECULAR )
		// Read specular texture
		specularTexColour = tex2D( specularMap, _input.texCoord );

		DEPENDENT_CODE_START( useEnvironmentMap )
		DEPENDENT_CODE_END( useEnvironmentMap )

#endif
	DEPENDENT_CODE_ELSE( useSpecular )
#if defined( _3DSMAX_ ) || !defined( USE_SPECULAR )
		// No specular, so default the colour to ones (will be optimised out)
		specularTexColour = float4( 0.0f, 0.0f, 0.0f, 0.0f );

		globalSpecularFactorValue = 0.0f;
		minSpecPowerValue = 0.0f;
		maxSpecPowerValue = 0.0f;
#endif
	DEPENDENT_CODE_END( useSpecular )
	

  // Normalise interpolated vectors
	float3 TSnormal = normalize( _input.normal );
  float3 eye = normalize( _input.eye.xyz );
	float3 normal;

	// Do tangent space normal mapping if required
	DEPENDENT_CODE_START( useNormalMap )
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
		// Normalise the input tangent and binormal vectors
		float3 tangent = normalize( _input.tangent );
		float3 binormal = normalize( _input.binormal );

		// Fetch and decode the map normal
		float4 normalMapColour = tex2D( normalMap, _input.texCoord );
		float3 normalFromMap;

		normalFromMap.rgb = normalize( ( normalMapColour.rgb * 2.0f ) - 1.0f );

		// Perturb the tangent space normal by the normal map
		normal = ( TSnormal * normalFromMap.z ) + ( normalFromMap.x * binormal ) + ( normalFromMap.y * tangent );
		normal = normalize( normal );
#endif
	DEPENDENT_CODE_ELSE( useNormalMap )
#if defined( _3DSMAX_ ) || !defined( NORMAL_MAPPING )
		// No normal map, so use interpolated normal and constant specular strength
		normal = TSnormal;
#endif
	DEPENDENT_CODE_END( useNormalMap )

	// Calculate base colour
	float4 accumulatedColour = diffuseTexColour * _input.colour;

	// If environment mapping is switched on
	DEPENDENT_CODE_START( useEnvironmentMap )
#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
		// Calculate the reflection vector
		float3 objectSpaceNormal = mul( float4( normal, 0.0f ), worldI ).xyz;
		float3 objectSpaceEye = mul( float4( -eye, 0.0f ), worldI ).xyz;
		float3 reflectionVector = reflect( objectSpaceEye, objectSpaceNormal );

		// Fetch the environment map colour using the world coords vector
#ifdef _3DSMAX_
		float4 environmentTexColour = texCUBElod( environmentMap, float4( reflectionVector.xzy, MAX_ENVMAP_BIAS * ( 1.0f - specularTexColour.a ) ) );
#else
		float4 environmentTexColour = texCUBElod( environmentMap, float4( reflectionVector, MAX_ENVMAP_BIAS * ( 1.0f - specularTexColour.a ) ) );
#endif

		// Calculate envmap colour and add to diffuse
		accumulatedColour += specularTexColour * environmentTexColour * globalEMapFactor;
#endif
	DEPENDENT_CODE_END( useEnvironmentMap )

	// Perform lighting
	DO_PS_LIGHTING_CALCULATIONS( accumulatedColour , _input.eye.xyz )

	accumulatedColour.w = diffuseTexColour.w * _input.colour.w;
	_output.Colour = CalculateOutputPixel(accumulatedColour);

	return _output;
}

REMOVE_UNUSED_INTERPOLATORS
PSOUTPUT SkintestUnlitFragmentShader( PSINPUT_UNLIT _input )
{
	PSOUTPUT _output;

	// Read textures
	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord );

	// Calculate base colour
	float4 accumulatedColour = diffuseTexColour * _input.colour;

	_output.Colour = CalculateOutputPixel(accumulatedColour);

	return _output;
}

REMOVE_UNUSED_INTERPOLATORS
PSOUTPUT SkintestShadowGenFragmentShader( SHADOWGEN_PSINPUT _input )
{
	PSOUTPUT output;
#if !defined(NEEDS_SHADOW_COORDS)
	output.Colour=0;
#else	
	CALC_SHADOWMAP_DEPTH( output.Colour, _input.shadowCoord );
#endif

	return output;
}

REMOVE_UNUSED_INTERPOLATORS
PSOUTPUT SkintestZPrimeDOFFragmentShader( ZPRIMEDOF_VSOUTPUT _input )
{
	PSOUTPUT output;

	output.Colour = _input.coords.z / _input.coords.w;

	return output;
}

REMOVE_UNUSED_INTERPOLATORS
PSOUTPUT SkintestZPrimeFragmentShader( ZPRIMEDOF_VSOUTPUT _input )
{
	PSOUTPUT output;

	output.Colour = 0.0f;

	return output;
}

REMOVE_UNUSED_INTERPOLATORS
PSOUTPUT SkintestOverdrawFragmentShader( PSINPUT _input )
{
	PSOUTPUT output;

	output.Colour = 0.05f;

	return output;
}

REMOVE_UNUSED_INTERPOLATORS
PSOUTPUT SkintestLowDetailFragmentShader( PSINPUT_LD _input )
{
	PSOUTPUT _output;

#if !defined( _3DSMAX_ ) && !defined( USE_SPECULAR )
	// In the engine, specialisations which have no specular defined need to declare these constants
	// as the lighting macros at the end need them, and they're not declared anywhere else.
	float globalSpecularFactorValue;
	float minSpecPowerValue;
	float maxSpecPowerValue;
#else
	float globalSpecularFactorValue = globalSpecularFactor;
	float minSpecPowerValue = minSpecPower;
	float maxSpecPowerValue = maxSpecPower;
#endif

	// Read textures
	float4 diffuseTexColour = Tint_DoIfEnabled( tex2D( diffuseMap, _input.texCoord ), _input.texCoord, true );
	float4 specularTexColour = float4( 0.0f, 0.0f, 0.0f, 0.0f );

	// Calculate base colour
	float4 accumulatedColour = diffuseTexColour * _input.colour;

	// If environment mapping is switched on
	DEPENDENT_CODE_START( useEnvironmentMap )
#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
	  // Normalise interpolated vectors
		float3 TSnormal = normalize( _input.normal );
	  float3 eye = normalize( _input.eye.xyz );
		float3 normal = TSnormal;

		// Calculate the reflection vector
		float3 objectSpaceNormal = mul( float4( normal, 0.0f ), worldI ).xyz;
		float3 objectSpaceEye = mul( float4( -eye, 0.0f ), worldI ).xyz;
		float3 reflectionVector = reflect( objectSpaceEye, objectSpaceNormal );
		float4 environmentTexColour = texCUBE( environmentMap, reflectionVector );

		// Calculate envmap colour and add to diffuse
		accumulatedColour += environmentTexColour * globalEMapFactor;
#endif
	DEPENDENT_CODE_END( useEnvironmentMap )

	// Perform lighting (without specular ... low detail)
	globalSpecularFactorValue = 0.0f;

	accumulatedColour.w = diffuseTexColour.w * _input.colour.w;
	_output.Colour = CalculateLowDetailOutputPixel(accumulatedColour);

	return _output;
}


//-----------------------------------------------------------------------
//
// Technique(s)
//

technique skintest
<
	int maxbones = MAX_BONES;
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	bool skinOnSpu=true;

	string normalBehaviour			= "ERMB_RENDER";
	string normalTechnique			= "skintest";
	int    normalDeferredID		= 0;

	string zprimeBehaviour			= "ERMB_RENDER";
	string zprimeTechnique			= "_skintest_zprime";
	int    zprimeDeferredID	        = 0;

	string zprimeDOFBehaviour		= "ERMB_RENDER";
	string zprimeDOFTechnique		= "_skintest_zprime_dof";
	int    zprimeDOFDeferredID	= 0;

	string shadowGenBehaviour		= "ERMB_RENDER";
	string shadowGenTechnique		= "_skintest_shadowgen";
	int    shadowGenDeferredID	= 0;

	string overdrawBehaviour = "ERMB_RENDER";
	string overdrawTechnique = "_skintest_overdraw";
	int    overdrawDeferredID	= 0;

	string lowDetailBehaviour = "ERMB_RENDER";
	string lowDetailTechnique = "skintest_lowdetail";
	int    lowDetailDeferredID	= 0;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = true;
		bool AlphaBlendEnable = false;
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
#endif

#if defined (_PS3_)		
		VertexShader = compile sce_vp_rsx SkintestVertexShader();
		PixelShader = compile sce_fp_rsx SkintestFragmentShader();
#else		
		VertexShader = compile vs_3_0 SkintestVertexShader();
		PixelShader = compile ps_3_0 SkintestFragmentShader();
#endif		
	}
}


technique skintest_translucent
<
	int maxbones = MAX_BONES;
	bool supportsSpecialisedLighting = true;
  bool preservesGlobalState = true;
	bool skinOnSpu=true;

	string normalBehaviour			= "ERMB_RENDER";
	string normalTechnique			= "skintest_translucent";
	int    normalDeferredID			= 2;

	string zprimeBehaviour			= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour		= "ERMB_DONT_RENDER";
	string shadowGenBehaviour		= "ERMB_RENDER";
	string shadowGenTechnique		= "_skintest_shadowgen";
	int    shadowGenDeferredID	= 0;

	string overdrawBehaviour = "ERMB_RENDER";
	string overdrawTechnique = "_skintest_overdraw";
	int    overdrawDeferredID	= 0;

	string lowDetailBehaviour = "ERMB_RENDER";
	string lowDetailTechnique = "skintest_lowdetail";
	int    lowDetailDeferredID	= 2;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = false;
		bool AlphaBlendEnable = true;
		string SrcBlend = "SRCALPHA";
		string DestBlend = "INVSRCALPHA";
		string BlendOp = "ADD";
>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = false;
		AlphaBlendEnable = true;
		SrcBlend = SRCALPHA;
		DestBlend = INVSRCALPHA;
		BlendOp = ADD;
#endif

#if defined (_PS3_)		
		VertexShader = compile sce_vp_rsx SkintestVertexShader();
		PixelShader = compile sce_fp_rsx SkintestFragmentShader();
#else		
		VertexShader = compile vs_3_0 SkintestVertexShader();
		PixelShader = compile ps_3_0 SkintestFragmentShader();
#endif		
	}
}


technique skintest_additive
<
	int maxbones = MAX_BONES;
	bool supportsSpecialisedLighting = true;
	// false as we mess with cull mode
	bool preservesGlobalState = false;
	bool skinOnSpu=true;

	string normalBehaviour			= "ERMB_RENDER";
	string normalTechnique			= "skintest_additive";
	int    normalDeferredID			= 2;

	string zprimeBehaviour			= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour		= "ERMB_DONT_RENDER";
	string shadowGenBehaviour		= "ERMB_DONT_RENDER";

	string overdrawBehaviour = "ERMB_RENDER";
	string overdrawTechnique = "_skintest_overdraw";
	int    overdrawDeferredID	= 0;

	string lowDetailBehaviour = "ERMB_RENDER";
	string lowDetailTechnique = "skintest_lowdetail";
	int    lowDetailDeferredID	= 2;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = false;
		bool AlphaBlendEnable = true;
		string SrcBlend = "SRCALPHA";
		string DestBlend = "ONE";
		string BlendOp = "ADD";
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = false;
		AlphaBlendEnable = true;
		SrcBlend = SRCALPHA;
		DestBlend = ONE;
		BlendOp = ADD;
#endif

#if defined (_PS3_)		
		CullFaceEnable = false;
		VertexShader = compile sce_vp_rsx SkintestVertexShader();
		PixelShader = compile sce_fp_rsx SkintestFragmentShader();
#else	
		CullMode = None;	
		VertexShader = compile vs_3_0 SkintestVertexShader();
		PixelShader = compile ps_3_0 SkintestFragmentShader();
#endif		
	}
}

technique skintest_additive_unlit
<
	int maxbones = MAX_BONES;
	bool supportsSpecialisedLighting = true;
	// false as we mess with cull mode
	bool preservesGlobalState = false;
	bool skinOnSpu=true;

	string normalBehaviour			= "ERMB_RENDER";
	string normalTechnique			= "skintest_additive_unlit";
	int    normalDeferredID			= 2;

	string zprimeBehaviour			= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour		= "ERMB_DONT_RENDER";
	string shadowGenBehaviour		= "ERMB_DONT_RENDER";

	string overdrawBehaviour = "ERMB_RENDER";
	string overdrawTechnique = "_skintest_overdraw";
	int    overdrawDeferredID	= 0;

	string lowDetailBehaviour = "ERMB_RENDER";
	string lowDetailTechnique = "skintest_lowdetail";
	int    lowDetailDeferredID	= 2;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = false;
		bool AlphaBlendEnable = true;
		string SrcBlend = "SRCALPHA";
		string DestBlend = "ONE";
		string BlendOp = "ADD";
		
		
>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = false;
		AlphaBlendEnable = true;
		SrcBlend = SRCALPHA;
		DestBlend = ONE;
		BlendOp = ADD;
#endif

#if defined (_PS3_)		
		CullFaceEnable = false;
		VertexShader = compile sce_vp_rsx SkintestUnlitVertexShader();
		PixelShader = compile sce_fp_rsx SkintestUnlitFragmentShader();
#else	
		CullMode = None;	
		VertexShader = compile vs_3_0 SkintestUnlitVertexShader();
		PixelShader = compile ps_3_0 SkintestUnlitFragmentShader();
#endif		
	}
}

technique skintest_alphatest
<
	int maxbones = MAX_BONES;
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	bool skinOnSpu=true;

	string normalBehaviour			= "ERMB_RENDER";
	string normalTechnique			= "skintest_alphatest";
	int    normalDeferredID			= 0;

	string zprimeBehaviour			= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour		= "ERMB_DONT_RENDER";
	string shadowGenBehaviour		= "ERMB_DONT_RENDER";

	string overdrawBehaviour = "ERMB_RENDER";
	string overdrawTechnique = "_skintest_overdraw";
	int    overdrawDeferredID	= 0;

	string lowDetailBehaviour = "ERMB_RENDER";
	string lowDetailTechnique = "skintest_lowdetail";
	int    lowDetailDeferredID	= 0;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool	ZEnable = true;
		bool	ZWriteEnable = true;
		bool	AlphaBlendEnable = false;
		bool AlphaTestEnable = true;
		int AlphaRef = 128;
		string AlphaFunc = "GreaterEqual";
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
		AlphaTestEnable = true;
		AlphaRef = 128;
		AlphaFunc = GreaterEqual;
#endif

#if defined (_PS3_)		
		VertexShader = compile sce_vp_rsx SkintestVertexShader();
		PixelShader = compile sce_fp_rsx SkintestFragmentShader();
#else		
		VertexShader = compile vs_3_0 SkintestVertexShader();
		PixelShader = compile ps_3_0 SkintestFragmentShader();
#endif		
	}
}

technique skintest_lowdetail
<
	int maxbones = MAX_BONES;
	bool preservesGlobalState = true;
	bool skinOnSpu=true;
	
	string normalBehaviour			= "ERMB_RENDER";
	string normalTechnique			= "skintest_lowdetail";
	int    normalDeferredID		= 0;

	string zprimeBehaviour			= "ERMB_RENDER";
	string zprimeTechnique			= "_skintest_zprime";
	int    zprimeDeferredID	        = 0;

	string zprimeDOFBehaviour		= "ERMB_RENDER";
	string zprimeDOFTechnique		= "_skintest_zprime_dof";
	int    zprimeDOFDeferredID	= 0;

	string shadowGenBehaviour		= "ERMB_RENDER";
	string shadowGenTechnique		= "_skintest_shadowgen";
	int    shadowGenDeferredID	= 0;

	string overdrawBehaviour = "ERMB_RENDER";
	string overdrawTechnique = "_skintest_overdraw";
	int    overdrawDeferredID	= 0;

	string lowDetailBehaviour = "ERMB_RENDER";
	string lowDetailTechnique = "skintest_lowdetail";
	int    lowDetailDeferredID	= 0;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = true;
		bool AlphaBlendEnable = false;
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
#endif

#if defined (_PS3_)		
		VertexShader = compile sce_vp_rsx SkintestLowDetailVertexShader();
		PixelShader = compile sce_fp_rsx SkintestLowDetailFragmentShader();
#else		
		VertexShader = compile vs_3_0 SkintestLowDetailVertexShader();
		PixelShader = compile ps_3_0 SkintestLowDetailFragmentShader();
#endif		
	}
}



technique _skintest_shadowgen
<
	int maxbones = MAX_BONES;
	bool preservesGlobalState = true;
	bool skinOnSpu=true;
>
{
	pass Pass0
	{
		AlphaTestEnable = false;
	
#if defined (_PS3_)		
		VertexShader = compile sce_vp_rsx SkintestShadowGenVertexShader();
		PixelShader = compile sce_fp_rsx SkintestShadowGenFragmentShader();
#else		
	#if defined(_XBOX) && !defined(SHADOWMAP_COLOUR)
		VertexShader = compile vs_3_0 SkintestShadowGenVertexShader();
		PixelShader = null;
	#else
		VertexShader = compile vs_3_0 SkintestShadowGenVertexShader();
		PixelShader = compile ps_3_0 SkintestShadowGenFragmentShader();
	#endif
#endif		
	}
}

technique _skintest_zprime_dof
<
	int maxbones = MAX_BONES;
	bool preservesGlobalState = true;
	bool skinOnSpu=true;
>
{
	pass Pass0
	{

#if defined (_PS3_)		
		VertexShader = compile sce_vp_rsx SkintestZPrimeDOFVertexShader();
		PixelShader = compile sce_fp_rsx SkintestZPrimeDOFFragmentShader();
#else		
		VertexShader = compile vs_3_0 SkintestZPrimeDOFVertexShader();
		PixelShader = compile ps_3_0 SkintestZPrimeDOFFragmentShader();
#endif		
	}
}

technique _skintest_zprime
<
	int maxbones = MAX_BONES;
	bool skinOnSpu=true;
>
{
	pass Pass0
	{
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = true;

#if defined (_PS3_)
		BlendEquation=int(FuncAdd);
		BlendFunc=int2(Zero, One);
		VertexShader = compile sce_vp_rsx SkintestZPrimeDOFVertexShader();
		PixelShader = compile sce_fp_rsx SkintestZPrimeFragmentShader();
#else		
		SrcBlend = ZERO;
		DestBlend = ONE;
		BlendOp = ADD;
		VertexShader = compile vs_3_0 SkintestZPrimeDOFVertexShader();
		PixelShader = compile ps_3_0 SkintestZPrimeFragmentShader();
#endif		
	}
}


technique _skintest_overdraw
<
	int maxbones = MAX_BONES;
	bool skinOnSpu=true;
>
{
	pass Pass0
	{
		ZEnable = false;
		ZWriteEnable = false;
		AlphaBlendEnable = true;

#if defined (_PS3_)
		BlendEquation=int(FuncAdd);
		BlendFunc=int2(One, One);
		VertexShader = compile sce_vp_rsx SkintestVertexShader();
		PixelShader = compile sce_fp_rsx SkintestOverdrawFragmentShader();
#else		
		SrcBlend = ONE;
		DestBlend = ONE;
		BlendOp = ADD;
		VertexShader = compile vs_3_0 SkintestVertexShader();
		PixelShader = compile ps_3_0 SkintestOverdrawFragmentShader();
#endif		
	}
}
